#version 330
#extension GL_EXT_gpu_shader4 : enable
// Terrain by Perlin NoiseMod01.fsh by battal
//https://www.shadertoy.com/view/MdtGRM
// Licence CC0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

#define iTime u_Elapsed*0.314159  //*0.1666
#define iResolution u_WindowSize

//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)
//#define iMouse vec4(512,256,180,120)
uniform sampler2D iChannel0;
uniform sampler2D iChannel1;
uniform sampler2D iChannel2;
uniform sampler2D iChannel3;
vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}
#define texture2D texture2D_Fract

/* ----- Constants ----- */

#define PI 3.14159265359

#define AUTO_REPEAT 256.0
#define TERRAIN_OCTAVES 5

#define FAR_PLANE 100.0

#define MIN_T 0.5
#define MAX_T 50.0

#define EPS 0.001


/* ----- Specification ----- */

/* Noise functions */
float hash(float x, float y);
float fade(float t);

int get_gradient(float xi, float yi, int repeat);
float get_dot(int gradient, float x, float y);

float noise(float x, float y, int repeat);
float octave_noise(float x, float y, int repeat, float frequency, float max_amplitude, float persistence);

/* Distance fields and ray marching */
float terrain(vec3 p);
float sphere(vec3 p);

float castRay(vec3 eye, vec3 ray);

/* Lighting */
float get_shadow(vec3 terrain_p, vec3 light_ray);
vec4 get_terrain_color(vec3 eye, vec3 ray, float t, vec3 sun_dir);

vec4 get_fog_color(vec3 ray, vec3 sun_dir);
float get_fog_amount(vec3 eye, vec3 ray, float t);
float get_fog_amount_const(float t);

vec4 get_lighting(vec3 eye, vec3 ray, float t);

/* Transformation */
mat4 inverse_view_matrix(vec3 look, vec3 up, vec3 pos);
mat4 inverse_projection_matrix(float width_angle, float height_angle, float far);


/* ----- Functions ----- */

/* Pseudo-random number generator */
float hash(float x, float y) {
    return fract(sin(floor(x) * 127.1 + floor(y) * 311.7) * 43758.5453123);
}

/* Interpolation function */
float fade(float t)
{
	return t * t * t * (t * (t * 6.0 - 15.0) + 10.0);
}

/* Integer representation of the gradient */
int get_gradient(float xi, float yi, int repeat)
{
    /* Take modulus to wrap overflowed numbers */
    if(repeat > 0)
    {
    	xi = mod(xi, float(repeat));
    	yi = mod(xi, float(repeat));
    }
    
    xi = mod(xi, AUTO_REPEAT);
    yi = mod(yi, AUTO_REPEAT);
    
    /* Random gradient in last four bits */
    float val = hash(xi, yi);
	return int(mod(val * 256.0, 256.0));
}

/* Dot product between gradient and location vector */
float get_dot(int gradient, float x, float y)
{
	int grad_bits = int(mod(float(gradient), 4.0));
    
    if(grad_bits == 0x0)
    {
    	return x + y;
    }
    else if(grad_bits == 0x1)
    {
    	return x - y;
    }
    else if(grad_bits == 0x2)
    {
    	return -x + y;
    }
    else if(grad_bits == 0x3)
    {
    	return -x - y;
    }
}

/* Perlin noise function */
float noise(float x, float y, int repeat)
{   
    /* Get inputs */
    if(repeat > 0)
    {
    	x = mod(x, float(repeat));
        y = mod(y, float(repeat));
    }
    
    
    /* Closest integral coordinates */
    float xi = mod(floor(x), AUTO_REPEAT);
    float yi = mod(floor(y), AUTO_REPEAT);
    
    /* Coordinates within unit box */
    float xf = fract(x);
    float yf = fract(y);
    
    /* Interpolation factors */
    float u = fade(xf);
    float v = fade(yf);
    
    /* Calculate gradients */
    int aa = get_gradient(xi, yi, repeat);
    int ab = get_gradient(xi + 1.0, yi, repeat);
    int ba = get_gradient(xi, yi + 1.0, repeat);
    int bb = get_gradient(xi + 1.0, yi + 1.0, repeat);
    
    /* Calculate influence */
    float g1 = get_dot(aa, xf, yf);
    float g2 = get_dot(ab, xf - 1.0, yf);
    float g3 = get_dot(ba, xf, yf - 1.0);
    float g4 = get_dot(bb, xf - 1.0, yf - 1.0);
    
    return mix(mix(g1, g2, u), mix(g3, g4, u), v);
}

/* Perlin noise with octaves, adjustable frequency and amplitude */
float octave_noise(float x, float y, int repeat, float frequency, float max_amplitude, float persistence)
{
    float total = 0.0;
    float amplitude = max_amplitude;
    
    for(int i = 0; i < TERRAIN_OCTAVES; i++)
    {
        /* Accumulate noise */
        total += amplitude * noise(x * frequency, y * frequency, repeat);
        
        /* Update frequency, amplitude */
        frequency = frequency * 2.0;
        amplitude = amplitude * persistence;
    }
    
    return total;
}

/* Terrain height function */
float terrain(vec3 p)
{
	return octave_noise(p.x, p.z, 0, 1.0 / 4.0, 1.5, 0.4);
}

/* Sphere distance function */
float sphere(vec3 p)
{
	return length(p - vec3(0.0, 1.0, -10.0)) - 0.5;
}

/* Ray march, return time of intersection */
float castRay(vec3 eye, vec3 ray)
{   
    float t = MIN_T;
    
    vec3 p = eye + t * ray;
    float h;
    
    for(int i = 0; i < 400; i++)
    {
        /* Get height to terrain */
        h = p.y - terrain(p);
        
        /* Return t if close enough */
        if(h < 0.001)
        {
        	return t;
        }
        
        if(t > MAX_T)
            return -1.0;
        
        /* Decrease precision */
        t += h * 0.75;
        
        /* Calculate next point */
        p = eye + ray * t;
    }
    
    return -1.0;
}


/* Get shadow from terrain point and light ray to point*/
float get_shadow(vec3 op, vec3 light_ray)
{
	float t = MIN_T;
    
    vec3 p = op + light_ray * t;
    float h;
    float shadow = 1.0;
    
    for(int i = 0; i < 400; i++)
    {
    	h = p.y - terrain(p);
        
        if(h < 0.001)
        {
            return 0.0;
        }
        
        if(t > MAX_T)
        {
            return shadow;
        }
        
        shadow = min(shadow, 10.0 * h / t);
        
        t += h * 0.75;
        p = op + light_ray * t;
    }
    
    return shadow;
}

/* Get terrain color */
vec4 get_terrain_color(vec3 eye, vec3 ray, float t, vec3 sun_dir)
{    
    float shadow = get_shadow(eye + ray * t, -sun_dir);
    
    /* Base terrain color (brown) */
    vec3 base_color = vec3((150.0 / 256.0),
                            (126.0 / 256.0),
                            (109.0 / 256.0));
    
    /* Normal */
    vec3 p = eye + ray * t;
    vec3 normal = normalize(vec3(terrain(p + vec3(-EPS, 0.0, 0.0)) - terrain(p + vec3(EPS, 0.0, 0.0)),
                       2.0 * EPS,
                       terrain(p + vec3(0.0, 0.0, -EPS)) - terrain(p + vec3(0.0, 0.0, EPS))));
    
    /* Ambient lighting */
    float ka = 0.2;
    vec3 ambient_color = base_color;
        
    /* Diffuse lighting */
    float kd = 0.6;
    float diffuse_dot = dot(normal, -sun_dir);
    vec3 diffuse_color = base_color;
    
    /* Specular lighting */
    float ks = 0.2;
    vec3 specular_color = base_color;
    
    return vec4(ka * ambient_color + kd * shadow * diffuse_dot * diffuse_color, 1.0);
}

/* Get fog color */
vec4 get_fog_color(vec3 ray, vec3 sun_dir)
{
    vec4 blue_fog = vec4(0.5, 0.6, 0.7, 1.0);
    vec4 yellow_fog = vec4(1.0, 0.9, 0.7, 1.0);
    
    float sun_amount = max(dot(ray, -sun_dir), 0.0);
    return mix(blue_fog, yellow_fog, pow(sun_amount, 8.0));	
}

/* Non-constant fog density (varying with height) */
float get_fog_amount(vec3 eye, vec3 ray, float t)
{
    float a = 0.09;
    float b = 1.0;
	return a * exp(-b * eye.y) * (1.0 - exp(-b * ray.y * t)) / (b * ray.y);
}

/* Constant fog density */
float get_fog_amount_const(float t)
{
    float b = 0.05;
	return 1.0 - exp(-b * t);
}

/* Get lighting for pixel */
vec4 get_lighting(vec3 eye, vec3 ray, float t)
{
    vec3 sun_dir = normalize(vec3(1.0, -0.3, -1.0));
    vec4 terrain_color = get_terrain_color(eye, ray, t, sun_dir);
    vec4 fog_color = get_fog_color(ray, sun_dir);
    
    return mix(terrain_color, fog_color, get_fog_amount(eye, ray, t));
}

/* Inverse of the view matrix */
mat4 inverse_view_matrix(vec3 look, vec3 up, vec3 pos)
{
	vec3 w = -normalize(look);
    vec3 v = normalize(up - w * dot(up, w));
    vec3 u = cross(v, w);
    return mat4(
    	u.x, v.x, w.x, 0.0,
        u.y, v.y, w.y, 0.0,
       	u.z, v.z, w.z, 0.0,
        0.0, 0.0, 0.0, 1.0
    );
}

/* Inverse projection matrix */
mat4 inverse_projection_matrix(float width_angle, float height_angle, float far)
{
    return mat4(
    	far * tan(width_angle / 2.0), 0.0, 0.0, 0.0,
        0.0, far * tan(height_angle / 2.0), 0.0, 0.0,
        0.0, 0.0, far, 0.0,
        0.0, 0.0, 0.0, 1.0
    );
}
void main (void)
//void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    /* Set up camera */
 	vec3 look = normalize(vec3(0.0, 0.01, -1.0));
    vec3 up = vec3(0.0, 1.0, 0.0);
    vec3 worldCameraPos = vec3(0.0, 3.0, 0.0);

    /* Get point on film plane */
	vec2 uv = gl_FragCoord.xy / iResolution.xy;
	vec3 normalizedFilmPoint = vec3(2.0 * uv.x - 1.0, uv.y * 2.0 - 1.0, -1.0);
    normalizedFilmPoint.x *= iResolution.x / iResolution.y;

    /* Transform to world space */
    mat4 ivm = inverse_view_matrix(look, up, worldCameraPos);
    mat4 ipm = inverse_projection_matrix(PI / 4.0, PI / 4.0, FAR_PLANE);
    vec3 filmPoint = vec3(ivm * ipm * vec4(normalizedFilmPoint, 1.0));
    
    /* Get ray */
    vec3 eye = vec3(worldCameraPos.xy, -iTime);
    vec3 ray = normalize(filmPoint - worldCameraPos);
    
    /* Ray march */
    float t = castRay(eye, ray);
    
    /* Lighting */
    if(t > 0.0)
    {
    	gl_FragColor = get_lighting(eye, ray, t);
    }
    else
 
    {
    	gl_FragColor = vec4(0.0, 0.0, 0.0, 0.0);
        gl_FragColor.a = length(gl_FragColor.rgb);
    }    
}